﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using NVCC.Models;
using NVCC.WebUI.Infrastructure;
using System.Web.ModelBinding;
using System.ComponentModel.DataAnnotations;
using System.Configuration;
using System.IO;
using iText.Kernel.Pdf;

namespace NVCC.WebUI.Models
{
    [Serializable]
    public class PatientProfileViewModel
    {
        private static readonly int MaxTotalSizeOfAttachedPdfs = int.Parse(ConfigurationManager.AppSettings["MaxTotalSizeOfAttachedPdfs"]);
        private static readonly int MaxSizeOfAttachedPdf = int.Parse(ConfigurationManager.AppSettings["MaxSizeOfAttachedPdf"]);
        private static readonly int MaxNumberOfAttachedPdfs = int.Parse(ConfigurationManager.AppSettings["MaxNumberOfAttachedPdfs"]);

        private string id;
        [BindNever]
        public string ID
        {
            get
            {
                return id;
            }
        }
        private void resetID()
        {
            id = null;
            id = String.Concat(PatientProfile.Patient.PatientSid, '-', this.GetHashCode().ToString("X8"));
        }
        public string PatientIcn { get; set; }
        public string UserName { get; set; }
        
        [AllowHtml]
        public PatientProfile PatientProfile { get; set; }
        public ReportOptionsModel ReportOptions { get; set; }
        public List<bool> AppointmentsSelected { get; set; }
        public Int32? AuthorizationSelected { get; set; }
        public Int32? ConsultSelected { get; set; }
        public List<bool> ProblemDiagnosesSelected { get; set; }
        public List<bool> RadiologyReportsSelected { get; set; }
        //public List<bool> pathologyReportsSelected { get; set; }
        public List<bool> NotesSelected { get; set; }
        public String AdditionalLabs { get; set; }
        [AllowHtml]
        public String ProgressNotes { get; set; }
        [AllowHtml]
        public String AdditionalRecords { get; set; }
        [AllowHtml]
        public String Aod { get; set; }
        [AllowHtml]
        public String AdditionalAuthorizations { get; set; }
        [AllowHtml]
        public String AdditionalConsults { get; set; }
        [AllowHtml]
        public String AdditionalRadiology { get; set; }

        //public string AdditionalPathology { get; set; }
        [AllowHtml]
        public String AdditionalOrders { get; set; }
        [BindNever]
        public IList<AuthorizationViewModel> AuthorizationViewModels { get; set; }
        [BindNever]
        public IList<ConsultViewModel> ConsultViewModels { get; set; }
        public string ReferralType { get; set; }

        [Required(ErrorMessage = "Please select a document type")]
        public string DocType { get; set; }
        public IEnumerable<SelectListItem> Types { get; set; }

        public List<string> Others { get; set; }

        public List<string> DisclosureDates { get; set; }

        public Disclosure DisclosureInfo { get; set; }
        public AdditionalObject AdditionalObject { get; set; }

        public IList<AdditionalObject> AdditionalObjects { get; set; }

        
        public List<bool> VistAImagingStudiesSelected { get; set; }

        public int StatusID { get; set; }
        public IEnumerable<SelectListItem> Statuses { get; set; }


        public List<HttpPostedFileBase> PdfsToAttach { get; set; }
        public List<UploadedFile> AttachedPdfs { get; set; }

        public StationInfo StationInfo { get; set; }

        public bool IncludeVistAImaging { get; set; }

        public PatientProfileViewModel() { }
        public PatientProfileViewModel(PatientProfile patientProfile)
        {
            AuthorizationViewModels = new List<AuthorizationViewModel>();
            ConsultViewModels = new List<ConsultViewModel>();
            PatientProfile = patientProfile;
            PatientIcn = patientProfile.Patient.PatientIcn;
            UserName = PatientProfile.UserInfo.Name;
            ReportOptions = new ReportOptionsModel();
            

            AppointmentsSelected = PatientProfile.Appointments == null ? new List<bool>() : Enumerable.Repeat(false, PatientProfile.Appointments.Count()).ToList();
            ProblemDiagnosesSelected = PatientProfile.ProblemDiagnoses == null ? new List<bool>() : Enumerable.Repeat(true, PatientProfile.ProblemDiagnoses.Count()).ToList();
            RadiologyReportsSelected = PatientProfile.RadiologyReports == null ? new List<bool>() : Enumerable.Repeat(false, PatientProfile.RadiologyReports.Count()).ToList();
            NotesSelected = PatientProfile.Notes == null ? new List<bool>() : Enumerable.Repeat(false, PatientProfile.Notes.Count()).ToList();
            //pathologyReportsSelected = patientProfile.PathologyReports == null ? new List<bool>() : Enumerable.Repeat(false, patientProfile.PathologyReports.Count()).ToList();
            
            PopulateViewHelpers();
        }
        /// <summary>
        /// Fills the parts of PatientProfileViewModel which are transforms of PatientProfile
        /// </summary>
        /// <remarks>
        /// The PatientProfileViewModel object contains some members which are translations of the
        /// data which is in the PatientProfile, specifically, the AuthorizationViewModels which is
        /// a translation of PatientProfile.Authorizations and ConsultViewModels which is a
        /// transformation of PatientProfile.MdwsConsults. When the PatientProfile is assigned
        /// or restored, theses need to be (re)initialized as well. Note that the selection members
        /// of the models do not need to be restored. They either come from the form submission or
        /// are created when the object is first created.
        /// </remarks>
        private void PopulateViewHelpers()
        {
            AuthorizationViewModels = PatientProfile.Authorizations.Select(auth => new AuthorizationViewModel(auth)).ToList();
            ConsultViewModels = PatientProfile.Consults.Select(cons => new ConsultViewModel(cons)).ToList();
        }
        public bool RefreshPatientProfile()
        {
            if (PatientIcn == null)
            {
                // not enough information to extract or rebuild PatientProfile (need at least ICN)
                return false;
            }
            if (PatientProfile == null || PatientProfile.Patient == null || PatientProfile.Patient.PatientSid == 0)
            {
                // need to refresh
                // have an ICN, so can refresh
                // Get it from the session object, if exists
                if (HttpContextManager.Current.Session != null)
                {
                    PatientProfile pp = HttpContextManager.Current.Session[PatientIcn] as PatientProfile;
                    if (pp != null)
                    {
                        PatientProfile = pp;
                        UserName = PatientProfile.UserInfo.Name;
                        PopulateViewHelpers();
                        return true;
                    }
                    else
                    {
                        // want to re-query the patient profile, which is something like the following (pulled from HomeController and PatientProfileController), but it is mising all the repository and service objects.
                        //Patient patient = _patientRepository.GetPatient(viewModel.PatientSSN, viewModel.Station);
                        //PatientProfile patientProfile = await _patientProfileService.GetPatientProfileAsync(patient, userIen);
                        //HttpContextManager.Current.Session["patientIcn"] = patientProfile;
                        //PatientProfile = patientProfile;
                        //return true;
                        return false;
                    }
                }
            }
            else
            {
                // no need to refresh.
                return true;
            }
            return false;
        }

        /// <summary>
        /// Stores the PatientProfileViewModel in the Session.
        /// </summary>
        /// <remarks>
        /// Recomputes a (potentially) new ID and uses that as the key in the Session collection. That new ID is stored both in the object (before it is stored in the Session) and returned from the method.
        /// </remarks>
        /// <returns>a string which is the key which can be used to look up the object in Session at a later time.</returns>
        public string cacheInSession()
        {
            PdfsToAttach = null;
            resetID();
            if (HttpContextManager.Current.Session != null)
            {                
                HttpContextManager.Current.Session[ID] = this;
                return ID;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// Gets the associated PatientProfileViewModel from the Session object based on the ID
        /// </summary>
        /// <remarks>
        /// The compliment of <c>cacheInSession()</c>. This will get the object associated with the handle that <c>cacheInSession()</c> returns.</remarks>
        /// <param name="ID">The key of the PatientProfileViewModel to be retrieved. Value from <c>cacheInSession()</c>.</param>
        /// <returns>The assocated PatientProfileViewModel object</returns>
        public static PatientProfileViewModel retrieveFromSession(string ID)
        {
            if (string.IsNullOrEmpty(ID)) return null;
            if (HttpContextManager.Current.Session == null) return null;
            return HttpContextManager.Current.Session[ID] as PatientProfileViewModel;
        }

        public bool AttachmentsArePdfs(List<HttpPostedFileBase> pdfsToAttach = null)
        {
            if (pdfsToAttach == null)
            {
                pdfsToAttach = PdfsToAttach;
            }
            const string pdfType = "application/pdf";
            if (pdfsToAttach != null)
            {
                foreach (var file in pdfsToAttach)
                {
                    if (file.ContentType != pdfType)
                    {
                        return false;
                    }
                }
            }
            return true;
        }

        public bool TotalSizeOfAttachedPdfsIsWithinLimit(List<HttpPostedFileBase> pdfsToAttach = null)
        {
            if (pdfsToAttach == null)
            {
                pdfsToAttach = PdfsToAttach;
            }

            int totalSize = 0;

            if (pdfsToAttach != null)
            {
                foreach (var pdf in pdfsToAttach)
                {
                    if (pdf != null)
                    {
                        totalSize += pdf.ContentLength;
                    }
                }
            }

            if (AttachedPdfs != null)
            {
                foreach (var pdf in AttachedPdfs)
                {
                    if (pdf != null)
                    {
                        totalSize += pdf.Size;
                    }
                }
            }

            return totalSize <= MaxTotalSizeOfAttachedPdfs;
        }

        public bool SizeOfEachPdfIsWithinLimit(List<HttpPostedFileBase> pdfsToAttach = null)
        {
            if (pdfsToAttach == null)
            {
                pdfsToAttach = PdfsToAttach;
            }
            if (pdfsToAttach != null)
            {
                foreach (var pdf in pdfsToAttach)
                {
                    if (pdf != null && pdf.ContentLength > MaxSizeOfAttachedPdf)
                    {
                        return false;
                    }
                }
            }
            return true;
        }

        public bool NumberOfPdfsIsWithinLimit(List<HttpPostedFileBase> pdfsToAttach = null)
        {
            if (pdfsToAttach == null)
            {
                pdfsToAttach = PdfsToAttach;
            }
            int pdfCount = 0;
            if (pdfsToAttach != null)
            {
                pdfCount += pdfsToAttach.Count();
            }
            if (AttachedPdfs != null)
            {
                pdfCount += AttachedPdfs.Count();
            }

            return pdfCount <= MaxNumberOfAttachedPdfs;
        }

        public void AttachPdfs(List<HttpPostedFileBase> pdfsToAttach = null)
        {
            if (pdfsToAttach == null)
            {
                pdfsToAttach = PdfsToAttach;
            }
            if (pdfsToAttach != null && pdfsToAttach.Any())
            {
                AttachedPdfs = AttachedPdfs ?? new List<UploadedFile>();

                foreach (var pdf in pdfsToAttach)
                {
                    AttachedPdfs.Add(new UploadedFile(pdf));
                }

                PdfsToAttach = null;
            }
        }

        public void RemoveSelectedAttachedPdfs()
        {
            if (AttachedPdfs != null)
            {
                AttachedPdfs.RemoveAll(p => p == null);

                var pdfsToRemove = new List<UploadedFile>();

                foreach (var pdf in AttachedPdfs)
                {
                    if (pdf.Remove)
                    {
                        try
                        {
                            File.Delete(pdf.FullPath);
                        }
                        catch
                        {
                            ///Not sure what to do in this catch block. Basically,
                            ///I want execution to continue even if Delete fails.
                        }
                        pdfsToRemove.Add(pdf);
                    }
                }
                foreach (var pdf in pdfsToRemove)
                {
                    AttachedPdfs.Remove(pdf);
                }
            }
        }

        public bool AttachedPdfsAreReadable()
        {
            var result = true;
            if (AttachedPdfs != null)
            {
                foreach (var pdf in AttachedPdfs)
                {
                    try
                    {
                        var pdfReader = new PdfReader(pdf.FullPath);
                    }
                    catch
                    {
                        pdf.Remove = true;
                        result = false;
                    }
                }
            }
            return result;
        }
    }
}